<?php
/*
Plugin Name: Library Value Calculator
Description: This plugin displays a library value calculator for public libraries on your WordPress site.
Version: 3.0
Author: Steve Gregory, Colorado State Library; updated by Dana, January 2026
Author URI: https://www.colibraries.org
License: Please contact the Colorado State Library for licensing information.
*/

if (!defined('WPINC')) {
  die;
}

register_activation_hook(__FILE__, 'library_value_calculator_install');
register_uninstall_hook(__FILE__, 'library_value_calculator_uninstall');
add_action('admin_menu', 'library_value_calculator_create_menu');
add_shortcode('library-value-calculator', 'library_value_calculator_shortcode');
add_action('admin_init', 'library_value_calculator_migrate_data');
add_filter('the_content', 'library_value_calculator_unautop_shortcode', 9);

// Strip <p> tags that wpautop adds around this shortcode output while preserving autop elsewhere
function library_value_calculator_unautop_shortcode($content)
{
  if (is_string($content) && has_shortcode($content, 'library-value-calculator')) {
    $content = shortcode_unautop($content);
  }

  return $content;
}

// Get default items array
function library_value_calculator_get_default_items()
{
  return array(
    'adbook' => array('cost' => '22', 'visible' => true, 'label' => 'Adult Books Borrowed per Month', 'admin_label' => 'Adult Books'),
    'yabook' => array('cost' => '18', 'visible' => true, 'label' => 'Young Adult Books Borrowed per Month', 'admin_label' => 'Young Adult Books'),
    'kidbook' => array('cost' => '16', 'visible' => true, 'label' => 'Children Books Borrowed per Month', 'admin_label' => 'Children Books'),
    'audbk' => array('cost' => '24', 'visible' => true, 'label' => 'Audio Books Borrowed per Month', 'admin_label' => 'Audio Books'),
    'ill' => array('cost' => '18', 'visible' => true, 'label' => 'Interlibrary Loans per Month', 'admin_label' => 'Interlibrary Loans'),
    'ebook' => array('cost' => '8', 'visible' => true, 'label' => 'Ebooks Borrowed per Month', 'admin_label' => 'Ebooks'),
    'magazine' => array('cost' => '7', 'visible' => true, 'label' => 'Magazines Borrowed per Month', 'admin_label' => 'Magazines'),
    'news' => array('cost' => '2', 'visible' => true, 'label' => 'Newspapers per Month', 'admin_label' => 'Newspapers'),
    'video' => array('cost' => '24', 'visible' => true, 'label' => 'DVDs Borrowed per Month', 'admin_label' => 'DVDs'),
    'cds' => array('cost' => '14', 'visible' => true, 'label' => 'CDs Borrowed per Month', 'admin_label' => 'CDs'),
    'music' => array('cost' => '13', 'visible' => true, 'label' => 'Music Downloaded per Month', 'admin_label' => 'Music Downloads'),
    'mtgrm' => array('cost' => '16', 'visible' => true, 'label' => 'Meeting Rooms Use (Hours per Month)', 'admin_label' => 'Meeting Rooms'),
    'adprog' => array('cost' => '34', 'visible' => true, 'label' => 'Adult Program/Class Attended per Month', 'admin_label' => 'Adult Programs'),
    'yaprog' => array('cost' => '33', 'visible' => true, 'label' => 'YA Program/Class Attended per Month', 'admin_label' => 'YA Programs'),
    'kidprog' => array('cost' => '33', 'visible' => true, 'label' => 'Children\'s Program/Class Attended per Month', 'admin_label' => 'Children\'s Programs'),
    'passes' => array('cost' => '20', 'visible' => true, 'label' => 'Museum and Cultural Passes Used per Month', 'admin_label' => 'Museum and Cultural Passes'),
    'compuse' => array('cost' => '27', 'visible' => true, 'label' => 'Computer Use (Hours per Month)', 'admin_label' => 'Computer Use'),
    'dbsearch' => array('cost' => '47', 'visible' => true, 'label' => 'Databases Used per Month (Number of Separate Databases)', 'admin_label' => 'Databases'),
    'refast' => array('cost' => '2', 'visible' => true, 'label' => 'Reference Questions Asked per Month', 'admin_label' => 'Reference Questions'),
  );
}

// Migrate old data structure to new structure
function library_value_calculator_migrate_data()
{
  $options = get_option('library_value_calculator_options');

  // Check if we need to migrate (old structure has roi_adbookCost, new has items array)
  if (isset($options['roi_adbookCost']) && !isset($options['items'])) {
    $items = library_value_calculator_get_default_items();

    // Override defaults with existing values
    $items['adbook']['cost'] = isset($options['roi_adbookCost']) ? $options['roi_adbookCost'] : $items['adbook']['cost'];
    $items['yabook']['cost'] = isset($options['roi_yabookCost']) ? $options['roi_yabookCost'] : $items['yabook']['cost'];
    $items['kidbook']['cost'] = isset($options['roi_kidbookCost']) ? $options['roi_kidbookCost'] : $items['kidbook']['cost'];
    $items['audbk']['cost'] = isset($options['roi_audbkCost']) ? $options['roi_audbkCost'] : $items['audbk']['cost'];
    $items['ill']['cost'] = isset($options['roi_illCost']) ? $options['roi_illCost'] : $items['ill']['cost'];
    $items['ebook']['cost'] = isset($options['roi_ebookCost']) ? $options['roi_ebookCost'] : $items['ebook']['cost'];
    $items['magazine']['cost'] = isset($options['roi_magazineCost']) ? $options['roi_magazineCost'] : $items['magazine']['cost'];
    $items['news']['cost'] = isset($options['roi_newsCost']) ? $options['roi_newsCost'] : $items['news']['cost'];
    $items['video']['cost'] = isset($options['roi_videoCost']) ? $options['roi_videoCost'] : $items['video']['cost'];
    $items['cds']['cost'] = isset($options['roi_cdsCost']) ? $options['roi_cdsCost'] : $items['cds']['cost'];
    $items['music']['cost'] = isset($options['roi_musicCost']) ? $options['roi_musicCost'] : $items['music']['cost'];
    $items['mtgrm']['cost'] = isset($options['roi_mtgrmCost']) ? $options['roi_mtgrmCost'] : $items['mtgrm']['cost'];
    $items['adprog']['cost'] = isset($options['roi_adprogCost']) ? $options['roi_adprogCost'] : $items['adprog']['cost'];
    $items['yaprog']['cost'] = isset($options['roi_yaprogCost']) ? $options['roi_yaprogCost'] : $items['yaprog']['cost'];
    $items['kidprog']['cost'] = isset($options['roi_kidprogCost']) ? $options['roi_kidprogCost'] : $items['kidprog']['cost'];
    $items['passes']['cost'] = isset($options['roi_passesCost']) ? $options['roi_passesCost'] : $items['passes']['cost'];
    $items['compuse']['cost'] = isset($options['roi_compuseCost']) ? $options['roi_compuseCost'] : $items['compuse']['cost'];
    $items['dbsearch']['cost'] = isset($options['roi_dbsearchCost']) ? $options['roi_dbsearchCost'] : $items['dbsearch']['cost'];
    $items['refast']['cost'] = isset($options['roi_refastCost']) ? $options['roi_refastCost'] : $items['refast']['cost'];

    // Keep the existing top-level settings and add the items array
    $options['items'] = $items;

    // Remove old keys
    unset($options['roi_adbookCost'], $options['roi_yabookCost'], $options['roi_kidbookCost']);
    unset($options['roi_audbkCost'], $options['roi_illCost'], $options['roi_ebookCost']);
    unset($options['roi_magazineCost'], $options['roi_newsCost'], $options['roi_videoCost']);
    unset($options['roi_cdsCost'], $options['roi_musicCost'], $options['roi_mtgrmCost']);
    unset($options['roi_adprogCost'], $options['roi_yaprogCost'], $options['roi_kidprogCost']);
    unset($options['roi_passesCost'], $options['roi_compuseCost'], $options['roi_dbsearchCost']);
    unset($options['roi_refastCost']);

    update_option('library_value_calculator_options', $options);
  }
}


// add stylesheet

function roi_admin_style_script()
{
  $plugin_url = plugin_dir_url(__FILE__);
  wp_enqueue_style('roi_admin_style_script', $plugin_url . "/admin/css/admin_styles.css");
  
  // Enqueue admin JavaScript
  wp_enqueue_script('roi_admin_script', $plugin_url . "/admin/js/admin-script.js", array(), '1.0', true);
  
  // Pass default values to JavaScript
  $default_items = library_value_calculator_get_default_items();
  $defaults_js = array();
  foreach ($default_items as $key => $item) {
    $defaults_js[$key] = $item['cost'];
  }
  wp_localize_script('roi_admin_script', 'roiCalculatorDefaults', $defaults_js);
}
add_action('admin_enqueue_scripts', 'roi_admin_style_script');

function library_value_calculator_install()
{
  // global $wp_version;
  // if (version_compare($wp_version, '6.0', '<')) { wp_die('This plugin requires WordPress version 6.0 or later.'); }

  // Declare default values for form variables
  $library_value_calculator_options = array(
    'roi_incomepercap' => '95.25',
    'roi_pagetitle' => 'Library Value Calculator for your library',
    'roi_lrsreportyear' => '2024',
    'items' => library_value_calculator_get_default_items(),
  );
  update_option('library_value_calculator_options', $library_value_calculator_options);
}

// Uninstall function - cleanup database on plugin removal
function library_value_calculator_uninstall()
{
  // Remove plugin options from database
  delete_option('library_value_calculator_options');

  // For multisite installations, remove from all sites
  if (is_multisite()) {
    global $wpdb;
    $blog_ids = $wpdb->get_col("SELECT blog_id FROM $wpdb->blogs");
    foreach ($blog_ids as $blog_id) {
      switch_to_blog($blog_id);
      delete_option('library_value_calculator_options');
      restore_current_blog();
    }
  }
}

// Create dashboard page

function library_value_calculator_create_settings_submenu()
{
  add_options_page('Library Value Calculator Settings Page', 'Library Value Calculator Settings', 'manage_options', 'library_value_calculator_settings_menu', 'library_value_calculator_settings_page');
  add_action('admin_init', 'library_value_calculator_register_settings');
}

function library_value_calculator_create_menu()
{
  add_menu_page('Library Value Calculator Settings Page', 'Library Value Calculator Settings', 'manage_options', 'library_value_calculator_settings_menu', 'library_value_calculator_settings_page');
  add_action('admin_init', 'library_value_calculator_register_settings');
}

function library_value_calculator_register_settings()
{
  //delete_option('library_value_calculator_options');
  register_setting('library_value_calculator_settings_group', 'library_value_calculator_options', array(
    'sanitize_callback' => 'library_value_calculator_sanitize_options'
  ));
}

function library_value_calculator_sanitize_options($input)
{
  $sanitized = array();

  // Sanitize top-level fields
  if (isset($input['roi_incomepercap'])) {
    $sanitized['roi_incomepercap'] = sanitize_text_field($input['roi_incomepercap']);
  }
  if (isset($input['roi_pagetitle'])) {
    $sanitized['roi_pagetitle'] = sanitize_text_field($input['roi_pagetitle']);
  }
  if (isset($input['roi_lrsreportyear'])) {
    $sanitized['roi_lrsreportyear'] = sanitize_text_field($input['roi_lrsreportyear']);
  }

  // Sanitize items array
  if (isset($input['items']) && is_array($input['items'])) {
    $sanitized['items'] = array();
    $allowed_keys = array_keys(library_value_calculator_get_default_items());

    foreach ($input['items'] as $key => $item) {
      // Only allow predefined item keys
      if (!in_array($key, $allowed_keys)) {
        continue;
      }

      $sanitized['items'][$key] = array();

      // Sanitize cost - must be numeric
      if (isset($item['cost'])) {
        $sanitized['items'][$key]['cost'] = sanitize_text_field($item['cost']);
      }

      // Sanitize label
      if (isset($item['label'])) {
        $sanitized['items'][$key]['label'] = sanitize_text_field($item['label']);
      }

      // Sanitize admin_label
      if (isset($item['admin_label'])) {
        $sanitized['items'][$key]['admin_label'] = sanitize_text_field($item['admin_label']);
      }

      // Checkboxes that are unchecked don't send any value
      $sanitized['items'][$key]['visible'] = isset($item['visible']) && $item['visible'] == '1';
    }
  }

  return $sanitized;
}

function library_value_calculator_settings_page()
{
  $library_value_calculator_options = (array) get_option('library_value_calculator_options');
  $roi_incomepercap = esc_attr($library_value_calculator_options['roi_incomepercap']);
  $roi_pagetitle = esc_attr($library_value_calculator_options['roi_pagetitle']);
  $roi_lrsreportyear = esc_attr($library_value_calculator_options['roi_lrsreportyear']);
  $items = isset($library_value_calculator_options['items']) ? $library_value_calculator_options['items'] : array();

  // Merge saved items with defaults to preserve admin_label and other default values
  $defaults = library_value_calculator_get_default_items();
  if (empty($items)) {
    $items = $defaults;
  } else {
    // Merge each item with its defaults to ensure all fields are present
    foreach ($defaults as $key => $default_item) {
      if (isset($items[$key])) {
        $items[$key] = array_merge($default_item, $items[$key]);
      } else {
        $items[$key] = $default_item;
      }
    }
  }
  ?>
  <!-- The settings dashboard -->
  <div class="wrap lrsAdmin">
    <h1>Settings for Library Value Calculator</h1>
    <h2>Instructions</h2>
    <p>The costs fields below have all been pre-filled with general costs for items and services. You may change any of
      those costs to reflect the value you feel the items and services are worth.</p>
    <p>At a minimum, you should update the the Local Operating Revenue Per Capita field to reflect your library's funding.
      You can find this value on the <a
        href="https://www.lrs.org/public-library-annual-report-dashboards/revenues-and-expenditures-individual-libraries/"
        target="_blank">Colorado Public Library Local Revenue (2024) dashboard</a>. (Link will open in a new tab.) Be sure
      to select the value from the <strong>Local Per Capita revenue</strong> column.</p>
    <p>The prices are pre-filled based on estimates and calculations. You can view those <a
        href="https://docs.google.com/spreadsheets/d/1ggkF524e8771ssQMJ1zbKQhSDyRRLOfLmQ2GE2LSPu8/edit?gid=0#gid=0"
        target="_blank">here</a> with a bit of the thinking behind them. (Link will open in a new tab.)</p>
    <div style="max-width: 50%; border: 1px solid #000; padding: 12px; background: #fff;">
      <h2>How to use it</h2>
      <ol>
        <li>Create a new page.</li>
        <li>Copy and paste the shortcode below into the page content area or a text block if you are using the Block
          Editor.<p style="font-size: 1.2em;"><strong>&#91;library-value-calculator&#93;</strong></p></li>
      </ol>
      
    </div>

    <form method="post" action="options.php" id="roiForm" aria-labelledby="roiSettingsHeading">
      <h2 id="roiSettingsHeading">Customize it for your library</h2>
      <?php settings_fields('library_value_calculator_settings_group'); ?>
      <input type="text" size="80" id="library_value_calculator_options[roi_pagetitle]"
        name="library_value_calculator_options[roi_pagetitle]" value="<?php echo $roi_pagetitle; ?>" />
      <label for="library_value_calculator_options[roi_pagetitle]">This text will be displayed at the top of the Library ROI
        Calculator page.</label></br></br>

      <div><input type="number" step="0.01" id="library_value_calculator_options[roi_incompercap]"
          name="library_value_calculator_options[roi_incomepercap]" size="6" value="<?php echo $roi_incomepercap; ?>" />
        <label for="library_value_calculator_options[roi_incompercap]">Local Operating Revenue Per Capita: Look this up on
          the Revenues and Expenditures dashboard linked to earlier.</label><br><br>
      </div>
      <div><input type="number" step="1" size="6" id="library_value_calculator_options[roi_lrsreportyear]"
          name="library_value_calculator_options[roi_lrsreportyear]" value="<?php echo $roi_lrsreportyear; ?>" />
        <label for="library_value_calculator_options[roi_lrsreportyear]">LRS Report Year: Be sure to note the year of the
          revenue figure above.</label><br><br>
      </div>

      <table id="costsTable">
        <caption>
          <h2>Items and Services Costs</h2>
        </caption>
        <thead>
          <tr>
            <th scope="col">Show on Calculator</th>
            <th scope="col">Category and Estimated Cost</th>
          </tr>
        </thead>
        <tbody> <?php foreach ($items as $key => $item):
          $cost = isset($item['cost']) ? esc_attr($item['cost']) : '';
          $label = isset($item['label']) ? esc_attr($item['label']) : '';
          $admin_label = isset($item['admin_label']) ? esc_attr($item['admin_label']) : '';
          $visible = isset($item['visible']) ? $item['visible'] : true;
          $safe_key = esc_attr($key);
          ?>
            <tr>
              <td>
                <label for="library_value_calculator_options[items][<?php echo $safe_key; ?>][visible]"
                  aria-label="<?php echo $label; ?>"></label>
                <input type="checkbox" class="visibility-checkbox"
                  id="library_value_calculator_options[items][<?php echo $safe_key; ?>][visible]"
                  name="library_value_calculator_options[items][<?php echo $safe_key; ?>][visible]" value="1" <?php checked($visible, true); ?> />
              </td>
              <td>
                <label for="library_value_calculator_options[items][<?php echo $safe_key; ?>][cost]"><?php echo $admin_label; ?>:</label>
                <input type="number" step="0.01" size="6"
                  id="library_value_calculator_options[items][<?php echo $safe_key; ?>][cost]"
                  name="library_value_calculator_options[items][<?php echo $safe_key; ?>][cost]"
                  value="<?php echo $cost; ?>" />
                <input type="hidden" name="library_value_calculator_options[items][<?php echo $safe_key; ?>][label]"
                  value="<?php echo $label; ?>" />
                <input type="hidden" name="library_value_calculator_options[items][<?php echo $safe_key; ?>][admin_label]"
                  value="<?php echo $admin_label; ?>" />
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>

      </table>

      <div style="margin-top: 20px; margin-bottom: 10px;">
        <button type="button" class="button" id="toggleAllCheckboxes">Toggle All Visible</button>
        <button type="button" class="button" id="resetToDefaults">Reset All to Defaults</button>
        <span id="admin-action-status" class="screen-reader-text" role="status" aria-live="polite"></span>
      </div>

      <p class="submit"><input type="submit" class="button-primary" value="Save Changes" /></p>
    </form>
  </div>
  <?php
}

// The public form
//function library_value_calculator($content)
function library_value_calculator_shortcode($atts = array(), $content = null, $tag = '')
{
  // Don't output content during admin/editor requests (WPBakery, Gutenberg, etc.)
  // This prevents conflicts with page builders trying to parse the page structure
  if (is_admin()) {
    return '';
  }

 // if (!strstr($content, "[library-roi-calculator]"))
   // return $content;
  // Enqueue CSS only when shortcode is present
  $plugin_url = plugin_dir_url(__FILE__);
  wp_enqueue_style('library_value_calculator_style', $plugin_url . "/public/css/calculator-styles.css");

  $library_value_calculator_options = (array) get_option('library_value_calculator_options');
  $roi_incomepercap = $library_value_calculator_options['roi_incomepercap'];
  $roi_pagetitle = $library_value_calculator_options['roi_pagetitle'];
  $roi_lrsreportyear = $library_value_calculator_options['roi_lrsreportyear'];
  $items = isset($library_value_calculator_options['items']) ? $library_value_calculator_options['items'] : array();

  // If items is empty, initialize with defaults
  if (empty($items)) {
    $items = library_value_calculator_get_default_items();
  }

  // Build JavaScript variables for costs
  $js_costs = '';
  foreach ($items as $key => $item) {
    $cost = isset($item['cost']) ? $item['cost'] : '0';
    $safe_key = preg_replace('/[^a-z_]/', '', $key); // Only allow lowercase letters and underscores
    $safe_cost = esc_js($cost);
    $js_costs .= "\tvar {$safe_key}Cost = \"{$safe_cost}\";\n";
  }

  $libcalc = '<script type="text/javascript">' . "\n";
  $libcalc .= "\tvar incpercap = \"" . esc_js($roi_incomepercap) . "\";\n";
  $libcalc .= "\tvar lrsreportyear = \"" . esc_js($roi_lrsreportyear) . "\";\n";
  $libcalc .= "\tvar roipagetitle = \"" . esc_js($roi_pagetitle) . "\";\n";
  $libcalc .= $js_costs;
  $libcalc .= '</script>' . "\n\n";
  $libcalc .= '<div id="library_value_calculator">' . "\n";
  $libcalc .= "\t<div id=\"lvc-validation-alert\" role=\"status\" aria-live=\"polite\" aria-atomic=\"true\" class=\"sr-only\"></div>\n";
  $libcalc .= "\t<h2>" . esc_html($roi_pagetitle) . "</h2>\n";
  $libcalc .= '<form action="" method="post" name="calculator">' . "\n";
  $libcalc .= '    <input type="hidden" name="roi" id="roi" value="0" />' . "\n";
  $libcalc .= '    <input type="hidden" name="totalvalue" id="totalvalue" value="0" />' . "\n";
  $libcalc .= "\t<p style=\"margin-top: 35px;\">Enter the number of times you use the following library services each <strong>month</strong></p>\n";
  $libcalc .= "\t<table>\n";
  $libcalc .= "\t<tr>\n";
  $libcalc .= '          <th scope="col">Your Monthly Use</th>' . "\n";
  $libcalc .= '          <th scope="col">Library Services</th>' . "\n";
  $libcalc .= '          <th scope="col">Total Monthly Value</th>' . "\n";
  $libcalc .= '          <th scope="col">Total Annual Value</th>' . "\n";
  $libcalc .= '        </tr>' . "\n";

  // Build table rows for visible items
  foreach ($items as $key => $item) {
    $visible = isset($item['visible']) ? $item['visible'] : true;
    if (!$visible)
      continue;

    $label = isset($item['label']) ? esc_html($item['label']) : '';
    $input_name = $key;
    $result_id = esc_attr($key . 'Result');

    // Map keys to display names for JavaScript variables
    $display_key = $key;
    if ($key == 'adbook')
      $display_key = 'adbooks';
    elseif ($key == 'yabook')
      $display_key = 'yabooks';
    elseif ($key == 'kidbook')
      $display_key = 'kidbooks';
    elseif ($key == 'audbk')
      $display_key = 'audio';
    elseif ($key == 'mtgrm')
      $display_key = 'meeting';
    elseif ($key == 'adprog')
      $display_key = 'adprogram';
    elseif ($key == 'yaprog')
      $display_key = 'yaprogram';
    elseif ($key == 'kidprog')
      $display_key = 'kidprogram';
    elseif ($key == 'compuse')
      $display_key = 'computer';
    elseif ($key == 'dbsearch')
      $display_key = 'database';
    elseif ($key == 'refast')
      $display_key = 'reference';

    $safe_display_key = esc_attr($display_key);
    $label_attr = esc_attr($label);
    $annual_result_id = esc_attr($key . 'AnnualResult');

    $libcalc .= "\n\t<tr>\n";
    $libcalc .= "\t\t<td><input type=\"number\" name=\"{$safe_display_key}\" step=\"1\" min=\"0\" id=\"{$safe_display_key}\" size=\"8\" onkeypress=\"libraryValueCalculatorValidateKeypress(event)\" onchange=\"libraryValueCalculatorHandleChange(this)\" oninput=\"libraryValueCalculatorHandleInput(this)\" title=\"Enter whole numbers only\" /></td>\n";
    $libcalc .= "\t\t<td><label for=\"{$safe_display_key}\">{$label}</label></td>\n";
    $libcalc .= "\t\t<td class=\"lrsCurrency\"><span class=\"lrsCurrencySymbol\" aria-hidden=\"true\">$</span><label for=\"{$result_id}\">\n";
    $libcalc .= "\t\t\t<input type=\"text\" id=\"{$result_id}\" class=\"lrsCurrencyInput\" size=\"6\" value=\"0.00\" readonly aria-live=\"polite\" aria-label=\"{$label_attr} monthly total\" />\n";
	$libcalc .= "\t\t\t<span class=\"sr-only\">{$result_id}</span>\n";
    $libcalc .= "\t\t</label></td>\n";
    $libcalc .= "\t\t<td class=\"lrsCurrency\"><span class=\"lrsCurrencySymbol\" aria-hidden=\"true\">$</span><label for=\"{$annual_result_id}\">\n";
    $libcalc .= "\t\t\t<input type=\"text\" id=\"{$annual_result_id}\" class=\"lrsCurrencyInput\" size=\"6\" value=\"0.00\" readonly aria-live=\"polite\" aria-label=\"{$label_attr} annual total\" />\n";
	$libcalc .= "\t\t\t<span class=\"sr-only\">{$annual_result_id}</span>\n";
    $libcalc .= "\t\t</label></td>\n";
    $libcalc .= "\t</tr>";
  }

  $libcalc .= "\n\t\t\t\n";
  $libcalc .= "\t<tr>\n";
    $libcalc .= "\t<td><input type=\"reset\" value=\"Reset\" /></td>\n";
  $libcalc .= "\t<td><div style=\"font-weight: bold; color: #000030;\">\n";
  $libcalc .= "\tValue you receive from <span id=\"librarychoice\">your library</span>:\n";
  $libcalc .= "\t</div></td>\n";
  $libcalc .= "\t<td style=\"font-weight: bold; color: #000030; font-size: larger; text-align: left;\">$<span id=\"totalResult\">____</span></td>\n";
  $libcalc .= "\t<td style=\"font-weight: bold; color: #000030; font-size: larger; text-align: left;\">$<span id=\"totalAnnualResult\">____</span></td>\n";

  $libcalc .= "\t</tr>\n\t\n";
  $libcalc .= "\t</table>\n\t\n";
  $libcalc .= "<br />\n";
  $libcalc .= "        <p style=\"font-weight: bold; font-size: 120%;\">For every <span class=\"red\" style=\"color: #600020;\">$1.00</span> in taxes you invest in your library, you receive \n";
  $libcalc .= "\t\t<span class=\"red\" style=\"color: #600020;\">$</span><span id=\"personalvalue\" style=\"color: #600020;\">_____</span> of value in return.</p>\n\n";
  $libcalc .= "\t<p class=\"small\" style=\"margin-top: 40px;\">\n";
  $libcalc .= "\t\t<strong>Where did these numbers come from?</strong></p>\n";
  $libcalc .= "\t<p>Typical annual taxpayer contributions are determined from the library's local income per capita. In <script type=\"text/javascript\">document.write(lrsreportyear);</script>, the average taxpayer in your library's service area paid $<script type=\"text/javascript\">document.write(incpercap);</script> in taxes to support your library.</p>\n";
  $libcalc .= "\t\n";
  $libcalc .= "\t<p>View the <a href=\"https://www.lrs.org/public-library-annual-report-dashboards/revenues-and-expenditures-individual-libraries/\">Colorado Public Library Local Revenue (2024) dashboard</a>.</p>\n";
  $libcalc .= "\t\t\t\t\t\t\t\t\n";
  $libcalc .= "\t<div class=\"small\" id=\"yourresult\">\n";
  $libcalc .= "\t\t<p>Your personal return on investment is based on your responses and the typical annual tax contribution. You see a returned value of $<span id=\"yourvalue\">____</span> for every one dollar invested.\n";
  $libcalc .= "\t\t</p>\n";
  $libcalc .= "\t</div>\n";
  $libcalc .= "    <div class=\"small\" id=\"credits\">\n";
  $libcalc .= "        <p>This calculator is a modification of the Maine State Library calculator, developed by the\n";
  $libcalc .= "            Library Research Service Read more about the calculator on the <a href=\"https://www.lrs.org/\">LRS website</a>.\n";
  $libcalc .= "        </p>\n";
  $libcalc .= "\t\t<p>This project is made possible by a grant from the <a href=\"https://www.imls.gov/\">U.S. Institute of Museum and Library Services (IMLS).</a></p>\n";
  $libcalc .= "    </div>\n";
  $libcalc .= "\t</form>\n";
  $libcalc .= "</div>\n";

  // Build JavaScript calculation function
  $js_calculate = "function calculate() {\n";
  $js_calculate .= "\tvar monthinc = incpercap / 12;\n";
  $js_calculate .= "\tvar totalresult = 0;\n\n";

  foreach ($items as $key => $item) {
    $visible = isset($item['visible']) ? $item['visible'] : true;
    if (!$visible)
      continue;

    $label = isset($item['label']) ? $item['label'] : '';

    // Map keys to display names for JavaScript variables
    $display_key = $key;
    if ($key == 'adbook')
      $display_key = 'adbooks';
    elseif ($key == 'yabook')
      $display_key = 'yabooks';
    elseif ($key == 'kidbook')
      $display_key = 'kidbooks';
    elseif ($key == 'audbk')
      $display_key = 'audio';
    elseif ($key == 'mtgrm')
      $display_key = 'meeting';
    elseif ($key == 'adprog')
      $display_key = 'adprogram';
    elseif ($key == 'yaprog')
      $display_key = 'yaprogram';
    elseif ($key == 'kidprog')
      $display_key = 'kidprogram';
    elseif ($key == 'compuse')
      $display_key = 'computer';
    elseif ($key == 'dbsearch')
      $display_key = 'database';
    elseif ($key == 'refast')
      $display_key = 'reference';

    $safe_key = preg_replace('/[^a-z_]/', '', $key);
    $safe_display_key = preg_replace('/[^a-z_]/', '', $display_key);
    $result_id = $safe_key . 'Result';
    $annual_result_id = $safe_key . 'AnnualResult';
    $value_var = $safe_display_key . 'Value';
    $annual_value_var = $safe_display_key . 'AnnualValue';

    $js_calculate .= "\tvar {$value_var} = document.calculator.{$safe_display_key}.value * {$safe_key}Cost;\n";
    $js_calculate .= "\tvar {$annual_value_var} = {$value_var} * 12;\n";
    $js_calculate .= "\tdocument.getElementById(\"{$result_id}\").value = {$value_var}.toFixed(2);\n";
    $js_calculate .= "\tdocument.getElementById(\"{$annual_result_id}\").value = {$annual_value_var}.toFixed(2);\n";
    $js_calculate .= "\ttotalresult += {$value_var};\n\n";
  }

  $js_calculate .= "\tdocument.getElementById(\"totalResult\").innerHTML = totalresult.toFixed(2);\n";
  $js_calculate .= "\tvar totalannual = totalresult * 12;\n";
  $js_calculate .= "\tdocument.getElementById(\"totalAnnualResult\").innerHTML = totalannual.toFixed(2);\n\n";
  $js_calculate .= "\tvar personalvalue = totalresult / monthinc;\n";
  $js_calculate .= "\tdocument.getElementById(\"yourvalue\").innerHTML = personalvalue.toFixed(2);\n";
  $js_calculate .= "\tdocument.getElementById(\"personalvalue\").innerHTML = personalvalue.toFixed(2);\n\n";

  // Add validation alerts
  foreach ($items as $key => $item) {
    $visible = isset($item['visible']) ? $item['visible'] : true;
    if (!$visible)
      continue;

    $label = isset($item['label']) ? $item['label'] : '';

    // Map keys to display names for JavaScript variables
    $display_key = $key;
    if ($key == 'adbook')
      $display_key = 'adbooks';
    elseif ($key == 'yabook')
      $display_key = 'yabooks';
    elseif ($key == 'kidbook')
      $display_key = 'kidbooks';
    elseif ($key == 'audbk')
      $display_key = 'audio';
    elseif ($key == 'mtgrm')
      $display_key = 'meeting';
    elseif ($key == 'adprog')
      $display_key = 'adprogram';
    elseif ($key == 'yaprog')
      $display_key = 'yaprogram';
    elseif ($key == 'kidprog')
      $display_key = 'kidprogram';
    elseif ($key == 'compuse')
      $display_key = 'computer';
    elseif ($key == 'dbsearch')
      $display_key = 'database';
    elseif ($key == 'refast')
      $display_key = 'reference';

    $safe_display_key = preg_replace('/[^a-z_]/', '', $display_key);
    $value_var = $safe_display_key . 'Value';
    $alert_msg = esc_js("Your response for {$label} should be a number.");

    $js_calculate .= "\tif(isNaN({$value_var})) {alert('{$alert_msg}');}\n";
  }

  $js_calculate .= "\n\tvar totalvalue=totalresult.toFixed(2);\n";
  $js_calculate .= "\tvar roi=personalvalue.toFixed(2);\n";
  $js_calculate .= "}\n\n";
  $js_calculate .= "// Reset form handler to clear calculated values\n";
  $js_calculate .= "document.addEventListener('DOMContentLoaded', function() {\n";
  $js_calculate .= "\tvar form = document.forms['calculator'];\n";
  $js_calculate .= "\tif (form) {\n";
  $js_calculate .= "\t\tform.addEventListener('reset', function() {\n";
  $js_calculate .= "\t\t\tsetTimeout(function() {\n";
  
  // Add reset for all individual result fields
  foreach ($items as $key => $item) {
    $visible = isset($item['visible']) ? $item['visible'] : true;
    if (!$visible) continue;
    $safe_key = preg_replace('/[^a-z_]/', '', $key);
    $js_calculate .= "\t\t\t\tdocument.getElementById('{$safe_key}Result').value = '0.00';\n";
    $js_calculate .= "\t\t\t\tdocument.getElementById('{$safe_key}AnnualResult').value = '0.00';\n";
  }
  
  $js_calculate .= "\t\t\t\tdocument.getElementById('totalResult').innerHTML = '____';\n";
  $js_calculate .= "\t\t\t\tdocument.getElementById('totalAnnualResult').innerHTML = '____';\n";
  $js_calculate .= "\t\t\t\tdocument.getElementById('personalvalue').innerHTML = '_____';\n";
  $js_calculate .= "\t\t\t\tdocument.getElementById('yourvalue').innerHTML = '____';\n";
  $js_calculate .= "\t\t\t}, 0);\n";
  $js_calculate .= "\t\t});\n";
  $js_calculate .= "\t}\n";
  $js_calculate .= "});\n";

  $libcalc .= "\n<script type=\"text/javascript\">\n";
  $libcalc .= "function announceToScreenReader(message) {\n";
  $libcalc .= "\tvar alertDiv = document.getElementById('lvc-validation-alert');\n";
  $libcalc .= "\tif (alertDiv) {\n";
  $libcalc .= "\t\talertDiv.textContent = '';\n";
  $libcalc .= "\t\tsetTimeout(function() {\n";
  $libcalc .= "\t\t\talertDiv.textContent = message;\n";
  $libcalc .= "\t\t}, 100);\n";
  $libcalc .= "\t}\n";
  $libcalc .= "}\n\n";
  $libcalc .= "function libraryValueCalculatorValidateKeypress(event) {\n";
  $libcalc .= "\tvar key = String.fromCharCode(event.which);\n";
  $libcalc .= "\tif (/[eE.\\-+]/.test(key)) {\n";
  $libcalc .= "\t\tevent.preventDefault();\n";
  $libcalc .= "\t\tannounceToScreenReader('Please enter numbers only.');\n";
  $libcalc .= "\t\treturn false;\n";
  $libcalc .= "\t}\n";
  $libcalc .= "}\n\n";
  $libcalc .= "function libraryValueCalculatorHandleChange(input) {\n";
  $libcalc .= "\tvar value = input.value;\n";
  $libcalc .= "\tif (value === '' || /^\\d+$/.test(value)) {\n";
  $libcalc .= "\t\tcalculate();\n";
  $libcalc .= "\t\treturn;\n";
  $libcalc .= "\t}\n";
  $libcalc .= "\tannounceToScreenReader('Please enter whole numbers only.');\n";
  $libcalc .= "\tinput.value = '';\n";
  $libcalc .= "\tinput.focus();\n";
  $libcalc .= "\tcalculate();\n";
  $libcalc .= "}\n\n";
  $libcalc .= "function libraryValueCalculatorHandleInput(input) {\n";
  $libcalc .= "\tcalculate();\n";
  $libcalc .= "}\n\n";
  $libcalc .= $js_calculate;
  $libcalc .= "</script>\n\n";

  //$content = str_replace("[library-roi-calculator]", $libcalc, $content);
  //return $content;
  return $libcalc;
}